/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.daemon;

import java.util.Map;
import java.util.HashMap;

import java.net.InetAddress;
import java.net.MulticastSocket;
import java.net.DatagramPacket;

import org.apache.log4j.Logger;
import org.apache.log4j.Level;
import org.videolan.vlma.common.IVlData;
import org.videolan.vlma.common.medias.IVlMedia;

import java.io.IOException;
import java.io.InterruptedIOException;

/**
 * This Watcher direcly connects to the multicast groups to verify if some data
 * are broadcasted. If yes, it considers that the media is broadcasted.
 * 
 * @author MagSoft
 */
public class VlDirectMulticastWatcher implements IVlStreamWatcher {

    private IVlData data;
    
    private static final Logger logger = Logger.getLogger(VlDirectMulticastWatcher.class);
    
    /**
     * The thread used to wait.
     */
    private Thread waitThread;
    
    /**
     * The multicast socket timeout
     */
    private static int SOCKET_RECEIVE_TIMEOUT = 500;
    
    /**
     * The minimal lenght of the string to consider that the media is
     * broadcasted
     */
    private static int BUF_LENGHT_MIN = 300;
    
    /**
     * The time leaved to the multicast socket to get data.
     */
    private static int WAIT_SPAN = 500;
    
    /**
     * Is true when the socket try to receive data
     */
    private boolean isReceivingData = false;
    
    public Map<InetAddress, InetAddress> getStreams() {
        
        Map<InetAddress, InetAddress> retour = new HashMap<InetAddress, InetAddress>();
        
        try {
            
            /* Lopp on the media if check if they have a program */
            for (IVlMedia media : data.getMedias()) {
                if (media.getProgram() != null && media.getProgram().getIp() != null) { 
                    /* If yes, we can start the test */
                    /* Join the multicast group */
                    logger.log(Level.DEBUG, "Joining the multicast group " + media.getProgram().getIp() + " of " + media.getName() + ".");
                    MulticastSocket s = new MulticastSocket(1234);
                    s.setSoTimeout(SOCKET_RECEIVE_TIMEOUT);
                    s.joinGroup(media.getProgram().getIp());
                    byte[] buf = new byte[1024];
                    DatagramPacket recv = new DatagramPacket(buf, buf.length);
                    /* Try to receive some data */
                    try {
                        isReceivingData = true;
                        startWaitingForTheSocket();
                        long bufLenght = 0;
                        while (isReceivingData) {
                            s.receive(recv);
                            bufLenght += recv.getLength();
                            /*
                             * If the minimum data quantity has been receive,
                             * then ...
                             */
                            if (bufLenght >= BUF_LENGHT_MIN)
                            {
                                /* It is no more necessary to wait */
                                isReceivingData = false;
                                waitThread.interrupt();
                            }
                        }
                        
                        /*
                         * This is a dirty test but I haven't seen something
                         * else at this moment
                         */
                        if (bufLenght >= BUF_LENGHT_MIN) { /*
                                                             * If some data is
                                                             * receidev, then
                                                             * the stream must
                                                             * be broadcast
                                                             */
                            retour.put(media.getProgram().getIp(), media.getProgram().getPlayer());
                            logger.log(Level.DEBUG, "Some data has been received : " + bufLenght + " bytes.");
                        }
                        else {
                            logger.log(Level.DEBUG, "Not enough data received (" + bufLenght + " bytes) verifying the channel " + media.getName() + ".");
                        }
                    } catch (InterruptedIOException e) {
                        logger.log(Level.DEBUG, "Socket TimeOut verifying the channel " + media.getName() + ".");
                    }
                    /* Leave the group */
                    s.leaveGroup(media.getProgram().getIp());
                }
            }
            
        } catch (IOException e) {
            // Do nothing
        }
        
        return retour;
    }
    
    /**
     * This wait WAIT_SPAN miliseconds for the multicast socket to receive some
     * data
     */
    Runnable waiter = new Runnable() {
        public void run() {
            /* We start waiting */
            try {
                Thread.sleep(WAIT_SPAN);
                /* Now the socket must stop receiving data */
                isReceivingData = false;
            }
            catch (InterruptedException e) { }
        }
    };
    
    /**
     * Start the waiting thread
     */
    private synchronized void startWaitingForTheSocket() {
        waitThread = new Thread(waiter);
        waitThread.setName("waitThread");
        waitThread.start();
    }
    
    public void setData(IVlData data) {
        this.data = data;
    }

}
