/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 * 
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.common.medias;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * A media group. A media group is a set of media which are all available using
 * the same adapter.
 * 
 * @author SylV
 * 
 */
public class VlMediaGroup implements Comparable<VlMediaGroup> {

    /**
     * The medias which belong to the media group.
     */
    public List<IVlMedia> medias;

    /**
     * Compares groups according to their priority.
     * 
     * @param o the object to compare to
     * @return 1 if the group has a higher priority, 0 if both groups have the
     *         same priority and -1 otherwise.
     */
    public int compareTo(VlMediaGroup o) {
        return o.getPriority().compareTo(getPriority());
    }

    /**
     * VlMedia default constructor.
     */
    public VlMediaGroup() {
        medias = new ArrayList<IVlMedia>();
    }

    /**
     * Compares the media group to the specified object.
     * 
     * @return true if the media group are equal; false otherwise.
     */
    public boolean equals(Object o) {
        if (o.getClass().equals(VlMediaGroup.class)) {
            VlMediaGroup g = (VlMediaGroup) o;
            if ((!g.medias.isEmpty()) && (!this.medias.isEmpty())) {
                return g.medias.get(0).sameGroup(medias.get(0));
            }
        }
        return false;
    }

    /**
     * Gives the media group priority. It is the sum of the priorities of all
     * media that belong to this group.
     * 
     * @return the group priority
     */
    public Integer getPriority() {
        int p = 0;
        Iterator<IVlMedia> i = medias.iterator();
        while (i.hasNext()) {
            p += i.next().getProgram().getPriority();
        }
        return Integer.valueOf(p);
    }

    /**
     * Gives the true class of all IVlMedia instances in the group.
     * 
     * @return the class of the IVlMedia instances
     */
    @SuppressWarnings("unchecked")
    public Class getMediaClass() {
        if (!medias.isEmpty()) {
            return (Class) medias.get(0).getClass();
        }
        return null;
    }
}
